/*
 *  i2c-mn2ws.c i2c driver for Panasonic MN2WS series
 *    (C) Copyright 2008 Panasonic Corporation.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <linux/platform_device.h>
#include <asm/uaccess.h>

#include <linux/i2c.h>
#include <linux/i2c/i2c-mn2ws.h>
#include "i2c-mn2ws-internal.h"

#include <asm/io.h>
#include <asm/irq.h>
#include <mach/intctl-regs.h>
#include <mach/clock.h>

#include <linux/syscalls.h>
#include <linux/kthread.h>

//#define DBG_I2C

#undef DEBUG_I2C_MN2WS

#ifdef DEBUG_I2C_MN2WS
#define DEB1(fmt, args...) do { if (i2c_debug>=1) printk(fmt, ## args); } while(0)
#define DEB2(fmt, args...) do { if (i2c_debug>=2) printk(fmt, ## args); } while(0)
#define DEB3(fmt, args...) do { if (i2c_debug>=3) printk(fmt, ## args); } while(0)
#else	//DEBUG_I2C_MN2WS
#define DEB1(fmt, args...) do {} while(0)
#define DEB2(fmt, args...) do {} while(0)
#define DEB3(fmt, args...) do {} while(0)
#endif	//DEBUG_I2C_MN2WS

//static int i2c_debug = 0;
static int i2c_debug = 3;

#undef DEBUG_IO
//#define DEBUG_IO

#ifdef DEBUG_IO
static char *names[] = {
	"DTRM", "DREC", "MYAD", "CLK", "BRST"
	, "HOLD", "BSTS", "NOISE", "SETUP"
};
#endif

static void i2c_mn2ws_add_watch_timer( struct i2c_algo_mn2ws_data *mn2ws_adap );
static void i2c_mn2ws_del_watch_timer( struct i2c_algo_mn2ws_data *mn2ws_adap );
static void i2c_mn2ws_watch_timer_handler( unsigned long data );

static unsigned int i2c_mn2ws_get_busrate(struct i2c_algo_mn2ws_data *adap);
static void i2c_mn2ws_set_busrate(struct i2c_algo_mn2ws_data *adap, unsigned int rate);
static void i2c_mn2ws_write32(struct i2c_algo_mn2ws_data *adap, unsigned long reg, unsigned long val);
static int i2c_mn2ws_write_cmd(struct i2c_algo_mn2ws_data *adap, unsigned long reg, unsigned long val, int timeout);
static unsigned long i2c_mn2ws_read32(struct i2c_algo_mn2ws_data *adap, unsigned long reg);
static int __devexit i2c_mn2ws_remove( struct platform_device *pdev );
static int i2c_mn2ws_suspend( struct platform_device *pdev, pm_message_t state );
static int i2c_mn2ws_resume( struct platform_device *pdev );


/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		void i2c_mn2ws_stop(struct i2c_adapter *i2c_adap)
 *
 * DESCRIPTION	:
 *		Generate a stop condition on the i2c bus
 *		returns after the stop condition has been generated
 *
 * INPUT		:
 *		struct i2c_adapter *i2c_adap		Adapter Info.
 *
 * OUTPUT		:
 *		Nothing
 *
 * RETURN		:
 *		void
 *
#endif
 *****************************************************************************/
static void
i2c_mn2ws_stop(
	struct i2c_adapter *i2c_adap
)
{
	struct i2c_algo_mn2ws_data *mn2ws_adap;
	unsigned long dtrm;

	DEB3( "\n\n##** %s, %d, %s **##\n\n", __FILE__, __LINE__, __func__ );
	if (!i2c_adap)
	{
		printk(KERN_ERR "i2c-algo-mn2ws: Access to invalid pointer.\n");
		return;
	}

	mn2ws_adap = i2c_adap->algo_data;
	
#if defined(CONFIG_I2C_SLAVE)
	if( ( mn2ws_adap->exec_info.mode == I2C_MN2WS_EXEC_SLAVE_RECV ) ||
		( mn2ws_adap->exec_info.mode == I2C_MN2WS_EXEC_SLAVE_SEND ) )
	{
		return ;
	}
#endif //#if defined(CONFIG_I2C_SLAVE)
	
	dtrm = I2C_MN2WS_DTRM_STO | I2C_MN2WS_DTRM_ACK;	// Stop Condition

	DEB1("=== STOP\n");

	i2c_mn2ws_write32(mn2ws_adap, I2C_MN2WS_DTRM, dtrm);
}

/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		int i2c_mn2ws_start(struct i2c_adapter *i2c_adap, struct i2c_msg *msg)
 *
 * DESCRIPTION	:
 *
 * INPUT		:
 *		struct i2c_adapter *i2c_adap		Adapter Info.
 *		struct i2c_msg *msg					Message Info.
 *
 * OUTPUT		:
 *		Nothing
 *
 * RETURN		:
 *		int		Error Code
 *				0		Normal
 *				< 0		Error
 *
#endif
 *****************************************************************************/
static int
i2c_mn2ws_start(
	struct i2c_adapter *i2c_adap, struct i2c_msg *msg
)
{
	struct i2c_algo_mn2ws_exec_info *exec_info;
	struct i2c_algo_mn2ws_data *mn2ws_adap;
	unsigned short flags, nak_ok;
	unsigned long dtrm;
	unsigned char addr;
	int ret = 0;

	DEB3( "\n\n##** %s, %d, %s **##\n\n", __FILE__, __LINE__, __func__ );
	if ((!i2c_adap) || (!msg))
	{
		printk(KERN_ERR "i2c-algo-mn2ws: Access to invalid pointer.\n");
		return -ENODEV;
	}

	mn2ws_adap = i2c_adap->algo_data;
	exec_info = &(mn2ws_adap->exec_info);
	flags   = msg->flags;
	nak_ok  = flags & I2C_M_IGNORE_NAK;

	DEB1("=== START/ReSTART\n");

	addr = ( msg->addr << 1 );
	if (flags & I2C_M_RD)
	{
		addr |= 1;
	}
	if (flags & I2C_M_REV_DIR_ADDR)
	{
		addr ^= 1;
	}
	dtrm = I2C_MN2WS_DTRM_IRQEN					// IRQ Enable
		 | I2C_MN2WS_DTRM_STA | I2C_MN2WS_DTRM_ACK	// Start/ReStart Condition
		 | (I2C_MN2WS_DTRM_DATA & addr);			// Data

	DEB2("=== SLAVE ADDRESS %#04x+%c=%#04x\n",
		msg->addr, flags & I2C_M_RD ? 'R' : 'W', addr
	);

	ret = i2c_mn2ws_write_cmd(mn2ws_adap, I2C_MN2WS_DTRM, dtrm, i2c_adap->timeout);
	if( -EREMOTEIO == ret )
	{
		printk(KERN_ERR "i2c-algo-mn2ws: %s: write cmd error.(%d)\n", __func__, ret);
		i2c_mn2ws_stop(i2c_adap);
	}
	
	return ret;
}


/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		int i2c_mn2ws_sendbytes(struct i2c_adapter *i2c_adap, struct i2c_msg *msg)
 *
 * DESCRIPTION	:
 *		Send Data to i2c-bus par Byte
 *
 * INPUT		:
 *		struct i2c_adapter *i2c_adap		Adapter Info.
 *		struct i2c_msg *msg					Message Info.
 *
 * OUTPUT		:
 *		Nothing
 *
 * RETURN		:
 *		int		Error Code
 *				>= 0	Normal (Write Count)
 *				<  0	Error
 *
#endif
 *****************************************************************************/
static int
i2c_mn2ws_sendbytes( struct i2c_adapter *i2c_adap )
{
	struct i2c_algo_mn2ws_data *mn2ws_adap;
	struct i2c_algo_mn2ws_exec_info *exec_info;
	unsigned char *buf;
	int wrcount = 0;
	int ret = 0;
	unsigned long dtrm;

	DEB3( "\n\n##** %s, %d, %s **##\n\n", __FILE__, __LINE__, __func__ );
	if( !i2c_adap )
	{
		printk(KERN_ERR "i2c-algo-mn2ws: Access to invalid pointer.\n");
		return -ENODEV;
	}

	mn2ws_adap = i2c_adap->algo_data;
	exec_info = &(mn2ws_adap->exec_info);
	buf = exec_info->buf;

	if( exec_info->len > 0 )
	{
		dtrm = I2C_MN2WS_DTRM_IRQEN							/* IRQ Enable */
			 | I2C_MN2WS_DTRM_ACK							/* NOP */
			 | (I2C_MN2WS_DTRM_DATA & buf[exec_info->cnt]);	/* Send Data */

		ret = i2c_mn2ws_write_cmd( mn2ws_adap, I2C_MN2WS_DTRM, dtrm, i2c_adap->timeout );
		if( ret < 0 )
		{
			printk(KERN_ERR "i2c-algo-mn2ws: %s: write cmd error.(%d)\n", __func__, ret);
			i2c_mn2ws_stop( i2c_adap );
			return ret;		/* NAK or TIMEDOUT */
		}

		wrcount = exec_info->cnt;
	}

	return wrcount;
}

/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		int i2c_mn2ws_readbytes(
 *			struct i2c_adapter *i2c_adap, struct i2c_msg *msg
 *		)
 *
 * DESCRIPTION	:
 *		Setup ACK or NACK for next received byte and wait for it to arrive.
 *
 * INPUT		:
 *		struct i2c_adapter *i2c_adap		Adapter Info.
 *		struct i2c_msg *msg					Message Info.
 *
 * OUTPUT		:
 *		struct i2c_msg *msg
 *
 * RETURN		:
 *		int		Error Code
 *				>= 0	Normal (Read Count)
 *				<  0	Error
 *
#endif
 *****************************************************************************/
static inline int
i2c_mn2ws_readbytes( struct i2c_adapter *i2c_adap )
{
	struct i2c_algo_mn2ws_data *mn2ws_adap;
	struct i2c_algo_mn2ws_exec_info *exec_info;
	int rdcount = 0;
	int ret = 0;
	unsigned long dtrm;

	DEB3( "\n\n##** %s, %d, %s **##\n\n", __FILE__, __LINE__, __func__ );
	if( !i2c_adap )
	{
		printk(KERN_ERR "i2c-algo-mn2ws: Access to invalid pointer.\n");
		return -ENODEV;
	}

	mn2ws_adap = i2c_adap->algo_data;
	exec_info = &(mn2ws_adap->exec_info);

	if( exec_info->len > 0 )
	{
		if( (exec_info->len > 1) && !(exec_info->flags & I2C_M_NO_RD_ACK) )
		{
			dtrm = I2C_MN2WS_DTRM_IRQEN;	/* IRQ Enable, Send Ack */
		}
		else {
			dtrm = I2C_MN2WS_DTRM_IRQEN		/* IRQ Enable */
				 | I2C_MN2WS_DTRM_ACK;		/* NoAck */
		}

		ret = i2c_mn2ws_write_cmd( mn2ws_adap, I2C_MN2WS_DTRM, dtrm, i2c_adap->timeout );
		if( ret < 0 )
		{
			printk(KERN_ERR "i2c-algo-mn2ws: %s: write cmd error.(%d)\n", __func__, ret);
			i2c_mn2ws_stop( i2c_adap );
			return ret;
		}

		rdcount = exec_info->cnt;
	}

	return rdcount;
}

/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		int i2c_mn2ws_xfer(
 *			struct i2c_adapter *i2c_adap, struct i2c_msg msgs[], int num
 *		)
 *
 * DESCRIPTION	:
 *
 * INPUT		:
 *		struct i2c_adapter *i2c_adap		Adapter Info.
 *		struct i2c_msg msgs[]				Message Info. Array
 *		int num								Number of Message Info. Array
 *
 * OUTPUT		:
 *		struct i2c_msg msgs[]				Message Info. Array
 *
 * RETURN		:
 *		int		Error Code
 *				>= 0	Normal(Array Count)
 *				<  0	Error
 *
#endif
 *****************************************************************************/
static int i2c_mn2ws_xfer(
	struct i2c_adapter *i2c_adap,
	struct i2c_msg msgs[],
	int num
)
{
	int i, ret = 0;
	struct i2c_msg *pmsg;
	struct i2c_algo_mn2ws_data *mn2ws_adap;
	struct i2c_algo_mn2ws_exec_info *exec_info;

	#ifdef DBG_I2C
	{
		unsigned int cnt;
		
		printk( "%d: %s:\n", current->pid, __func__ );
		printk( "%d: %s: msg[0].len   = 0x%08X\n", current->pid, __func__, msgs[0].len );
		printk( "%d: %s: msg[0].addr  = 0x%08X\n", current->pid, __func__, msgs[0].addr );
		printk( "%d: %s: msg[0].flags = 0x%08X\n", current->pid, __func__, msgs[0].flags );
		for( cnt = 0; cnt < msgs[0].len; cnt++ )
		{
			printk( "%d: %s: msg[0].buf[%d] = 0x%08X\n", current->pid, __func__, cnt, msgs[0].buf[cnt] );
		}
		for( cnt = 1; cnt < num; cnt++ )
		{
			printk( "%d: %s: msg[%d].len   = 0x%08X\n", current->pid, __func__, cnt, msgs[cnt].len );
			printk( "%d: %s: msg[%d].addr  = 0x%08X\n", current->pid, __func__, cnt, msgs[cnt].addr );
			printk( "%d: %s: msg[%d].flags = 0x%08X\n", current->pid, __func__, cnt, msgs[cnt].flags );
		}
	}
	#endif	//DBG_I2C
	DEB3( "\n\n##** %s, %d, %s **##\n\n", __FILE__, __LINE__, __func__ );

#if defined(CONFIG_I2C_SLAVE)	
	if( i2c_adap->attr != I2C_ATTR_MASTER )
	{
		printk(KERN_ERR "i2c-algo-mn2ws: call master-ioctl from slave.\n");
		ret = -EPERM;
		goto L_END;
	}
#endif //defined(CONFIG_I2C_SLAVE)

	if ((!i2c_adap) || (!msgs))
	{
		printk(KERN_ERR "i2c-algo-mn2ws: Access to invalid pointer.\n");
		ret = -ENODEV;
		goto L_END;
	}
	mn2ws_adap = i2c_adap->algo_data;
	exec_info = &(mn2ws_adap->exec_info);

	DEB1("=== i2c_mn2ws_xfer\n");

	{
		unsigned long cnt;
		unsigned long drec;
		unsigned long bsts;
		
		for( cnt = 0; cnt < I2C_MN2WS_BUS_POLLING_CNT; cnt++ )
		{
			drec = i2c_mn2ws_read32( mn2ws_adap, I2C_MN2WS_DREC );
			if( drec & I2C_MN2WS_DREC_BB ){
				break;
			}
			cond_resched();
		}
		if( I2C_MN2WS_BUS_POLLING_CNT == cnt ){
			ret = -EAGAIN;
			goto L_END;
		}
		
		for( cnt = 0; cnt < I2C_MN2WS_BUS_POLLING_CNT; cnt++ )
		{
			bsts = i2c_mn2ws_read32( mn2ws_adap, I2C_MN2WS_BSTS );
			if( (I2C_MN2WS_BSTS_SDA & bsts) && (I2C_MN2WS_BSTS_SCL & bsts) ){
				break;
			}
			cond_resched();
		}
		if( I2C_MN2WS_BUS_POLLING_CNT == cnt ){
			ret = -EAGAIN;
			goto L_END;
		}
	}
	
	for (i = 0; i < num; ++i)
	{
		/* READ wait just after WRITE */
		if( i != 0 ) {
			if( (msgs[i].flags & I2C_M_RD) && ( !(msgs[i-1].flags & I2C_M_RD)) )
			{
				if( mn2ws_adap->wr2rd_wait_time != 0 )
				{
					unsigned long	waitjiffies;

					waitjiffies = ((unsigned long)mn2ws_adap->wr2rd_wait_time
					               + 1000 / HZ -1) * HZ / 1000 + 1;
					set_current_state( TASK_UNINTERRUPTIBLE );
					schedule_timeout( (long)waitjiffies );
				}
			}
		}

		pmsg = &msgs[i];
		exec_info->flags = pmsg->flags;		/* control flag */
		exec_info->len = pmsg->len;			/* data length */
		exec_info->buf = pmsg->buf;			/* data buffer pointer */
		exec_info->cnt = 0;					/* send/recv counter */

		if( !(pmsg->flags & I2C_M_NOSTART) )
		{
			// Start or ReStart Condition & Slave Address & R/W Command
			exec_info->mode = I2C_MN2WS_EXEC_START;		/* execution mode */
			ret = i2c_mn2ws_start( i2c_adap, pmsg );
			if( ret < 0 )
			{
				if( ret == -EREMOTEIO )
				{
					DEB2("=== i2c_mn2ws_xfer: NAK from device addr %2.2x msg #%d\n"
						, msgs[i].addr, i
					);
				}
				goto L_END;
			}
		}
		else if( pmsg->flags & I2C_M_RD )
		{
			/* read bytes into buffer*/
			exec_info->mode = I2C_MN2WS_EXEC_READ;		/* execution mode */
			ret = i2c_mn2ws_readbytes( i2c_adap );
			DEB2("=== i2c_mn2ws_xfer: read %d bytes.\n", ret);
			if( ret < 0 )
			{
				goto L_END;
			}
		}
		else {
			/* write bytes from buffer */
			exec_info->mode = I2C_MN2WS_EXEC_SEND;		/* execution mode */
			ret = i2c_mn2ws_sendbytes( i2c_adap );
			DEB2("=== i2c_mn2ws_xfer: wrote %d bytes.\n", ret);
			if( ret < 0 )
			{
				goto L_END;
			}
		}
	}
	DEB1("=== i2c_mn2ws_xfer: STOP CONDITION\n");
	i2c_mn2ws_stop(i2c_adap);	/* STOP CONDITION */

	ret = num;
L_END:
	#ifdef DBG_I2C
		if( 0 <= ret ){
			printk( "%d: %s: return = %d\n", current->pid, __func__, ret );
		}else{
			printk( "%d: %s: \x1b[31mERROR\x1b[30m: return = %d\n", current->pid, __func__, ret );
		}
	#endif	//DBG_I2C
	return ret;
}

#if defined(CONFIG_I2C_SLAVE)
/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		int i2c_mn2ws_slave_xfer(
 *			struct i2c_adapter *i2c_adap, struct i2c_msg *msg
 *		)
 *
 * DESCRIPTION	:
 *
 * INPUT		:
 *		struct i2c_adapter *i2c_adap		Adapter Info.
 *		struct i2c_msg *msgs				Message Info.
 *
 * OUTPUT		:
 *		struct i2c_msg *msgs				Message Info.
 *
 * RETURN		:
 *		int		Error Code
 *				>= 0	Normal(R/W data byte Count)
 *				<  0	Error
 *
#endif
 *****************************************************************************/
static int i2c_mn2ws_slave_xfer(
        struct i2c_adapter *i2c_adap,
        struct i2c_msg *msg
)
{
	int ret = 0;
	struct i2c_algo_mn2ws_data *mn2ws_adap;
	struct i2c_algo_mn2ws_exec_info *exec_info;

	DEB1("=== i2c_mn2ws_slave_xfer\n");

	#ifdef DBG_I2C
	{
		unsigned int cnt;
		
		printk( "%d: %s:\n", current->pid, __func__ );
		printk( "%d: %s: msg.len   = 0x%08X\n", current->pid, __func__, msg->len );
		printk( "%d: %s: msg.flags = 0x%08X\n", current->pid, __func__, msg->flags );
		for( cnt = 0; cnt < msgs.len; cnt++ )
		{
			printk( "%d: %s: msg.buf[%d] = 0x%08X\n", current->pid, __func__, cnt, msg->buf[cnt] );
		}
	}
	#endif	//DBG_I2C
	DEB3( "\n\n##** %s, %d, %s **##\n\n", __FILE__, __LINE__, __func__ );
	if (!i2c_adap)
	{
		printk(KERN_ERR "i2c-algo-mn2ws: Access to invalid pointer.\n");
		ret = -ENODEV;
		goto L_END;
	}
	
	mn2ws_adap = i2c_adap->algo_data;
	exec_info = &(mn2ws_adap->exec_info);
	
	exec_info->flags = msg->flags;		/* control flag */
	exec_info->len = msg->len;			/* data length */
	exec_info->buf = msg->buf;			/* data buffer pointer */
	exec_info->cnt = 0;					/* send/recv counter */
	exec_info->slave_cond = I2C_SLAVE_COND_CONTINUE;
	
	if( msg->flags & I2C_M_RD )
	{
		/* write bytes from buffer */
		if( exec_info->mode != I2C_MN2WS_EXEC_SLAVE_SEND )
		{
			printk(KERN_ERR "i2c-algo-mn2ws: illegal ioctl() transfer command.\n");
			ret = -EPERM;
			goto L_END;
		}
		
		ret = i2c_mn2ws_sendbytes( i2c_adap );
		DEB2("=== i2c_mn2ws_slave_xfer: wrote %d bytes.\n", ret);
		if( ret < 0 )
		{
			exec_info->mode = I2C_MN2WS_EXEC_IDLE;
			goto L_END;
		}
	}
	else {
		/* read bytes into buffer*/
		if( exec_info->mode != I2C_MN2WS_EXEC_SLAVE_RECV )
		{
			printk(KERN_ERR "i2c-algo-mn2ws: illegal ioctl() transfer command.\n");
			ret = -EPERM;
			goto L_END;
		}
		
		ret = i2c_mn2ws_readbytes( i2c_adap );
		DEB2("=== i2c_mn2ws_slave_xfer: read %d bytes.\n", ret);
		if( ret < 0 )
		{
			exec_info->mode = I2C_MN2WS_EXEC_IDLE;
			goto L_END;
		}
	}

	msg->cond = exec_info->slave_cond;
	
	switch( exec_info->slave_cond )
	{
		case I2C_SLAVE_COND_STOP:
			exec_info->mode = I2C_MN2WS_EXEC_IDLE;
			break;
		case I2C_SLAVE_COND_READ:
			exec_info->mode = I2C_MN2WS_EXEC_SLAVE_SEND;
			break;
		case I2C_SLAVE_COND_WRITE:
			exec_info->mode = I2C_MN2WS_EXEC_SLAVE_RECV;
			break;
		default:
			break;
	}
L_END:	
	#ifdef DBG_I2C
		if( 0 <= ret ){
			printk( "%d: %s: return = %d\n", current->pid, __func__, ret );
		}else{
			printk( "%d: %s: \x1b[31mERROR\x1b[30m: return = %d\n", current->pid, __func__, ret );
		}
	#endif	//DBG_I2C
	return ret;
}

/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		int i2c_mn2ws_slave_wait_req(
 *			struct i2c_adapter *i2c_adap
 *		)
 *
 * DESCRIPTION	:
 *
 * INPUT		:
 *		struct i2c_adapter *i2c_adap		Adapter Info.
 *
 * OUTPUT		:
  *
 * RETURN		:
 *		int		Error Code
 *				>= 0	Normal(request transfer mode)
 *				<  0	Error
 *
#endif
 *****************************************************************************/
static int i2c_mn2ws_slave_wait_req( struct i2c_adapter *adap )
{
	struct i2c_algo_mn2ws_data *mn2ws_adap;
	int ret;
	void *addr;
	unsigned long val;
	int irq;
	unsigned long flags;
	
	DEB1("=== i2c_mn2ws_slave_wait_req\n");

	if (!adap)
	{
		printk(KERN_ERR "i2c-algo-mn2ws: Access to invalid pointer.\n");
		ret = -ENODEV;
		goto L_END;
	}
	
	mn2ws_adap = adap->algo_data;

	if (!mn2ws_adap)
	{
		printk(KERN_ERR "i2c-mn2ws: Access to invalid pointer.\n");
		ret = -ENODEV;
		goto L_END;
	}
	
	spin_lock_irqsave(&(mn2ws_adap->lock), flags);
	if(mn2ws_adap->exec_info.mode != I2C_MN2WS_EXEC_IDLE)
	{
		spin_unlock_irqrestore(&(mn2ws_adap->lock), flags);
		printk(KERN_ERR "i2c-algo-mn2ws: i2c_mn2ws_slave_wait_req already called.\n");
		ret = -EPERM;
		goto L_END;
	}
	mn2ws_adap->exec_info.mode = I2C_MN2WS_EXEC_SLAVE_WAIT;
	spin_unlock_irqrestore(&(mn2ws_adap->lock), flags);

	
	irq = mn2ws_adap->irq;
	if (irq < 0)
	{
		printk(KERN_ERR "i2c-mn2ws: mn2ws_adap->irq < 0\n");
		mn2ws_adap->exec_info.mode = I2C_MN2WS_EXEC_IDLE;
		ret = -EAGAIN;
		goto L_END;
	}

	mn2ws_adap->irq_event = 0;
	mn2ws_adap->slave_event = 0;

	val = I2C_MN2WS_DTRM_IRQEN;						/* IRQ Enable */
	addr = mn2ws_adap->iomap_base + (I2C_MN2WS_DTRM & I2C_MN2WS_RMSK);
	writel(val, addr);

	/*
	 * Enable IRQ
	 */
	enable_irq(irq);

	if( wait_event_interruptible( mn2ws_adap->slave_wait, (mn2ws_adap->slave_event != 0) )){
		printk(KERN_ERR "i2c-mn2ws: wait_event_interruptible error.\n");
		mn2ws_adap->exec_info.mode = I2C_MN2WS_EXEC_IDLE;
		ret = -EINTR;
	}
	else
	{
		DEB2("=== i2c_mn2ws_slave_wait_req: %c\n",
			mn2ws_adap->slave_event & I2C_SLAVE_REQ_READ ? 'R' : 'W'
		);
		ret = mn2ws_adap->slave_event;
	}

	/* Disable IRQ */
	disable_irq(irq);

	mn2ws_adap->slave_event = 0;

L_END:
	return ret;
}
#endif /* CONFIG_I2C_SLAVE */

/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		u32 i2c_mn2ws_func(struct i2c_adapter *i2c_adap)
 *
 * DESCRIPTION	:
 *
 * INPUT		:
 *		struct i2c_adapter *i2c_adap		Adapter Info.
 *
 * OUTPUT		:
 *		Nothing
 *
 * RETURN		:
 *		u32
 *
#endif
 *****************************************************************************/
static u32 i2c_mn2ws_func(struct i2c_adapter *i2c_adap)
{
	u32 ret;
	
	#ifdef DBG_I2C
		printk( "%d: %s:\n", current->pid, __func__ );
	#endif	//DBG_I2C
	DEB1("=== i2c_mn2ws_func\n");
	DEB3( "\n\n##** %s, %d, %s **##\n\n", __FILE__, __LINE__, __func__ );
	ret = I2C_FUNC_I2C;
	#ifdef DBG_I2C
		if( 0 <= ret ){
			printk( "%d: %s: return = %d\n", current->pid, __func__, ret );
		}else{
			printk( "%d: %s: \x1b[31mERROR\x1b[30m: return = %d\n", current->pid, __func__, ret );
		}
	#endif	//DBG_I2C
	return ret;
}

/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		int i2c_mn2ws_init(struct i2c_algo_mn2ws_data *mn2ws_adap)
 *
 * DESCRIPTION	:
 *
 * INPUT		:
 *		struct i2c_algo_mn2ws_data *mn2ws_adap		Bus Info.
 *
 * OUTPUT		:
 *		Nothing
 *
 * RETURN		:
 *		int		Error Code
 *				0		Normal
 *				< 0		Error
 *
#endif
 *****************************************************************************/
static int
i2c_mn2ws_init(struct i2c_algo_mn2ws_data *mn2ws_adap)
{
	DEB3( "\n\n##** %s, %d, %s **##\n\n", __FILE__, __LINE__, __func__ );
	if (!mn2ws_adap)
	{
		printk(KERN_ERR "Access to invalid pointer.\n");
		return -ENODEV;
	}

	/*
	 * Set Bus Rate
	 */
	i2c_mn2ws_set_busrate(mn2ws_adap, mn2ws_adap->rate);

	/*
	 * Enable IIC bus (Release & Enable)
	 */
	i2c_mn2ws_write32(
		mn2ws_adap, I2C_MN2WS_BRST
		, I2C_MN2WS_BRST_FOEN | I2C_MN2WS_BRST_BRST
	);
#if defined(CONFIG_I2C_SLAVE)
	mn2ws_adap->slave_event = 0;
	mn2ws_adap->exec_info.mode = I2C_MN2WS_EXEC_IDLE;
	mn2ws_adap->exec_info.slave_cond = I2C_SLAVE_COND_STOP;
#endif //#if defined(CONFIG_I2C_SLAVE)
	
	return 0;
}

/******************************************************************************
 * Data
 *****************************************************************************/
static struct i2c_algorithm mn2ws_algo = {
	.master_xfer   = i2c_mn2ws_xfer,
#if defined(CONFIG_I2C_SLAVE)
	.slave_xfer    = i2c_mn2ws_slave_xfer,
	.wait_req      = i2c_mn2ws_slave_wait_req,
#endif //defined(CONFIG_I2C_SLAVE)
	.functionality = i2c_mn2ws_func,
};


/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		int i2c_mn2ws_add_bus(struct i2c_adapter *i2c_adap)
 *
 * DESCRIPTION	:
 *		registering functions to load algorithms at runtime 
 *
 * INPUT		:
 *		struct i2c_adapter *i2c_adap		Adapter Info.
 *
 * OUTPUT		:
 *		Nothing
 *
 * RETURN		:
 *		int		Error Code
 *				0		Normal
 *				< 0		Error
 *
#endif
 *****************************************************************************/
static int
i2c_mn2ws_add_bus(struct i2c_adapter *i2c_adap)
{
	struct i2c_algo_mn2ws_data *mn2ws_adap = i2c_adap->algo_data;
	int rval;

	DEB3( "\n\n##** %s, %d, %s **##\n\n", __FILE__, __LINE__, __func__ );
	/* register new adapter to i2c module... */
	i2c_adap->algo  = &mn2ws_algo;
#if defined(CONFIG_I2C_SLAVE)	
	if(i2c_adap->attr == I2C_ATTR_MASTER)
	{
		i2c_adap->timeout = HZ;	/* default values, should */
	}
	else
	{
		i2c_adap->timeout = I2C_SLAVE_TIMEOUT_VALUE;	/* default values, should */
	}
#else
	i2c_adap->timeout = HZ;	/* default values, should */
#endif //defined(CONFIG_I2C_SLAVE)

	i2c_adap->retries = 3;		/* be replaced by defines */

	rval = i2c_mn2ws_init(mn2ws_adap);

	if (!rval)
	{
		i2c_add_adapter(i2c_adap);
	}
	return rval;
}

/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		int i2c_mn2ws_del_bus(struct i2c_adapter *i2c_adap)
 *
 * DESCRIPTION	:
 *
 * INPUT		:
 *		struct i2c_adapter *i2c_adap		Adapter Info.
 *
 * OUTPUT		:
 *		Nothing
 *
 * RETURN		:
 *		int		Error Code
 *				0		Normal
 *				< 0		Error
 *
#endif
 *****************************************************************************/
int
i2c_mn2ws_del_bus(struct i2c_adapter *i2c_adap)
{
	DEB3( "\n\n##** %s, %d, %s **##\n\n", __FILE__, __LINE__, __func__ );
	return i2c_del_adapter(i2c_adap);
}

/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		int i2c_mn2ws_reset(
 *			struct i2c_algo_mn2ws_data *mn2ws_adap
 *			, struct i2c_busreset_ioctl_data *setting
 *		)
 *
 * DESCRIPTION	:
 *		Reset the i2c bus
 *
 * INPUT		:
 *		struct i2c_algo_mn2ws_data *mn2ws_adap		Bus Info.
 *		struct i2c_busreset_ioctl_data *setting				Reset Info.
 *
 * OUTPUT		:
 *		Nothing
 *
 * RETURN		:
 *		int		Error Code
 *				>= 0	Normal (Write Count)
 *				<  0	Error
 *
#endif
 *****************************************************************************/
static int
i2c_mn2ws_reset(
	struct i2c_algo_mn2ws_data *mn2ws_adap
	, struct i2c_busreset_ioctl_data *setting
)
{
	unsigned int interval, times;
#if defined(CONFIG_I2C_SLAVE)
	struct i2c_algo_mn2ws_exec_info *exec_info;
#endif //defined(CONFIG_I2C_SLAVE)
	
	DEB3( "\n\n##** %s, %d, %s **##\n\n", __FILE__, __LINE__, __func__ );
	if ((!mn2ws_adap) || (!setting))
	{
		return -EINVAL;
	}

	interval = setting->interval;
	times    = setting->times;
	if ((!interval) || (!times))
	{
		return times ? -EINVAL : 0;
	}

	while (times)
	{
		i2c_mn2ws_write32(mn2ws_adap, I2C_MN2WS_BRST, 0);
		udelay(interval);
		i2c_mn2ws_write32(mn2ws_adap, I2C_MN2WS_BRST, I2C_MN2WS_BRST_BRST);
		udelay(interval);
		--times;
	}

	/* Bus Enable */
	i2c_mn2ws_write32(
		mn2ws_adap, I2C_MN2WS_BRST
		, I2C_MN2WS_BRST_FOEN | I2C_MN2WS_BRST_BRST
	);
	
#if defined(CONFIG_I2C_SLAVE)
	exec_info = &(mn2ws_adap->exec_info);
	
	if(exec_info->mode == I2C_MN2WS_EXEC_SLAVE_WAIT)
	{
		mn2ws_adap->slave_event = -EAGAIN;
		wake_up_interruptible( &(mn2ws_adap->slave_wait) );
	}
	
	exec_info->mode = I2C_MN2WS_EXEC_IDLE;
#endif //defined(CONFIG_I2C_SLAVE)
	
	#ifdef DBG_I2C
		{
			unsigned long drec;
			unsigned long bsts;
			
			drec = i2c_mn2ws_read32( mn2ws_adap, I2C_MN2WS_DREC );
			if( !(drec & I2C_MN2WS_DREC_BB) ){
				printk( "%d: %s: \x1b[31mERROR\x1b[30m:\n", current->pid, __func__ );
			}
			bsts = i2c_mn2ws_read32( mn2ws_adap, I2C_MN2WS_BSTS );
			if( !(I2C_MN2WS_BSTS_SDA & bsts) || !(I2C_MN2WS_BSTS_SCL & bsts) ){
				printk( "%d: %s: \x1b[31mERROR\x1b[30m:\n", current->pid, __func__ );
			}
		}
	#endif	//DBG_I2C
	return 0;
}


static int i2c_mn2ws_reset2( struct i2c_algo_mn2ws_data *mn2ws_adap )
{
	unsigned long drec;
	unsigned long bsts;
	unsigned long timeout;
	#if defined(ISC_SLAVE) && defined(CONFIG_I2C_SLAVE)
		struct i2c_algo_mn2ws_exec_info *exec_info;
	#endif
	
	
	if( !mn2ws_adap ){
		return -EINVAL;
	}

	i2c_mn2ws_write32( mn2ws_adap, I2C_MN2WS_BRST, I2C_MN2WS_BRST_BRST );
	drec = i2c_mn2ws_read32( mn2ws_adap, I2C_MN2WS_DREC );
	if( 0x00004000 & drec ){
		i2c_mn2ws_write32( mn2ws_adap, I2C_MN2WS_DTRM, 0x00000200 );
		timeout = jiffies + HZ;
		while(1)
		{
			bsts = i2c_mn2ws_read32( mn2ws_adap, I2C_MN2WS_BSTS );
			if( (I2C_MN2WS_BSTS_SDA & bsts) && (I2C_MN2WS_BSTS_SCL & bsts) ){
				break;
			}
			if( time_after( jiffies, timeout ) ){
				bsts = i2c_mn2ws_read32( mn2ws_adap, I2C_MN2WS_BSTS );
				if( (I2C_MN2WS_BSTS_SDA & bsts) && (I2C_MN2WS_BSTS_SCL & bsts) ){
					break;
				}
				printk( "\n\n##** ERR : %s, %s, %d **##\n\n", __FILE__, __func__, __LINE__ );
				return -ETIMEDOUT;
			}
			msleep(1);
		}
	}
	
	i2c_mn2ws_write32( mn2ws_adap, I2C_MN2WS_BRST, 0 );
	i2c_mn2ws_write32( mn2ws_adap, I2C_MN2WS_BRST, I2C_MN2WS_BRST_BRST );
	timeout = jiffies + HZ;
	while(1)
	{
		bsts = i2c_mn2ws_read32( mn2ws_adap, I2C_MN2WS_BSTS );
		if( (I2C_MN2WS_BSTS_SDA & bsts) && (I2C_MN2WS_BSTS_SCL & bsts) ){
			break;
		}
		if( time_after( jiffies, timeout ) ){
			bsts = i2c_mn2ws_read32( mn2ws_adap, I2C_MN2WS_BSTS );
			if( (I2C_MN2WS_BSTS_SDA & bsts) && (I2C_MN2WS_BSTS_SCL & bsts) ){
				break;
			}
			printk( "\n\n##** ERR : %s, %s, %d **##\n\n", __FILE__, __func__, __LINE__ );
			return -ETIMEDOUT;
		}
		msleep(1);
	}
	
	i2c_mn2ws_write32( mn2ws_adap, I2C_MN2WS_BRST, I2C_MN2WS_BRST_FOEN | I2C_MN2WS_BRST_BRST );
	
	#if defined(ISC_SLAVE) && defined(CONFIG_I2C_SLAVE)
		exec_info = &mn2ws_adap->exec_info;
		
		if( exec_info->mode == I2C_MN2WS_EXEC_SLAVE_WAIT ){
			mn2ws_adap->slave_event = -EAGAIN;
			wake_up_interruptible( &(mn2ws_adap->slave_wait) );
		}
		
		exec_info->mode = I2C_MN2WS_EXEC_IDLE;
	#endif //#if defined(ISC_SLAVE)
	
	return 0;
}


/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		int i2c_mn2ws_control(
 *			struct i2c_adapter *i2c_adap, unsigned int cmd, unsigned long arg
 *		)
 *
 * DESCRIPTION	:
 *
 * INPUT		:
 *		struct i2c_adapter *i2c_adap		Adapter Info.
 *		unsigned int cmd					Command
 *		unsigned long arg					Command Arg
 *
 * OUTPUT		:
 *		unsigned long arg					Command Arg
 *
 * RETURN		:
 *		int		Error Code
 *				0		Normal
 *				< 0		Error
 *
#endif
 *****************************************************************************/
static int i2c_mn2ws_control(
	struct i2c_adapter *i2c_adap, unsigned int cmd, unsigned long arg
)
{
	struct i2c_algo_mn2ws_data *mn2ws_adap;
	int		status;

	DEB3( "\n\n##** %s, %d, %s **##\n\n", __FILE__, __LINE__, __func__ );
	if (!i2c_adap)
	{
		printk(KERN_ERR "i2c-algo-mn2ws: Access to invalid pointer.\n");
		return -ENODEV;
	}
	mn2ws_adap = i2c_adap->algo_data;

	DEB1("=== i2c_mn2ws_control\n");

	if (!arg)
	{
		return -EINVAL;
	}

	switch (cmd)
	{
	default:
		return -ENOIOCTLCMD;

	case I2C_BUSRESET:
		{
			struct i2c_busreset_ioctl_data	sBusresetData;
			
			i2c_lock_adapter( i2c_adap );
			#ifdef DBG_I2C
				printk( "%d: %s: BUSRESET\n", current->pid, __func__ );
			#endif	//DBG_I2C
			status = copy_from_user( &sBusresetData, (void *)arg, sizeof(struct i2c_busreset_ioctl_data) );
			if( status ){
				return -EFAULT;
			}
			status = i2c_mn2ws_reset( mn2ws_adap, &sBusresetData );
			#ifdef DBG_I2C
				if( 0 <= status ){
					printk( "%d: %s: return = %d\n", current->pid, __func__, status );
				}else{
					printk( "%d: %s: \x1b[31mERROR\x1b[30m: return = %d\n", current->pid, __func__, status );
				}
			#endif	//DBG_I2C
			i2c_unlock_adapter( i2c_adap );
			return status;
		}
	case I2C_BUSRESET2:
		{
			i2c_lock_adapter( i2c_adap );
			status = i2c_mn2ws_reset2( mn2ws_adap );
			i2c_unlock_adapter( i2c_adap );
			return status;
		}
	case I2C_SET_BUSRATE:
		if ((arg < 1) || (40 < arg))
		{
			return -EINVAL;
		}
		i2c_lock_adapter( i2c_adap );
		#ifdef DBG_I2C
			printk( "%d: %s: SET_BUSRATE\n", current->pid, __func__ );
		#endif	//DBG_I2C
		i2c_mn2ws_set_busrate(mn2ws_adap, arg);
		#ifdef DBG_I2C
			printk( "%d: %s: BUSRESET: return = 0\n", current->pid, __func__ );
		#endif	//DBG_I2C
		i2c_unlock_adapter( i2c_adap );
		break;
	case I2C_GET_BUSRATE:
		{
			unsigned int	busRate;
			
			i2c_lock_adapter( i2c_adap );
			#ifdef DBG_I2C
				printk( "%d: %s: SET_BUSRATE\n", current->pid, __func__ );
			#endif	//DBG_I2C
			busRate = i2c_mn2ws_get_busrate(mn2ws_adap);
			put_user( busRate, (unsigned int *)arg );
			#ifdef DBG_I2C
				printk( "%d: %s: BUSRESET: return = 0\n", current->pid, __func__ );
			#endif	//DBG_I2C
			i2c_unlock_adapter( i2c_adap );
			break;
		}
	}
	return 0;
}

/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		void i2c_mn2ws_write32(
 *			struct i2c_algo_mn2ws_data *, unsigned long, unsigned long
 *		)
 *
 * DESCRIPTION	:
 *		Write Register
 *
 * INPUT		:
 *		struct i2c_algo_mn2ws_data *adap	Bus Info.
 *		unsigned long reg						Address Offset
 *		unsigned long val						Write Value
 *
 * OUTPUT		:
 *		Nothing
 *
 * RETURN		:
 *		void
 *
#endif
 *****************************************************************************/
static void
i2c_mn2ws_write32(
	struct i2c_algo_mn2ws_data *adap, unsigned long reg, unsigned long val
)
{
	void *addr;
	unsigned long flags;

	DEB3( "\n\n##** %s, %d, %s **##\n\n", __FILE__, __LINE__, __func__ );
	
	if (!adap)
	{
		printk(KERN_ERR "i2c-mn2ws: Access to invalid pointer.\n");
		return;
	}
	spin_lock_irqsave(&(adap->lock), flags);
	addr = adap->iomap_base + (reg & I2C_MN2WS_RMSK);
	writel(val, addr);	/* iowrite32(val, addr); */
	spin_unlock_irqrestore(&(adap->lock), flags);

#ifdef DEBUG_IO
	DEB3("*** write %s at %#lx <= %#04x\n", names[reg/4], addr, val);
#endif
}

/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		unsigned long
 *			i2c_mn2ws_read32(struct i2c_algo_mn2ws_data *, unsigned long)
 *
 * DESCRIPTION	:
 *		Read Register
 *
 * INPUT		:
 *		struct i2c_algo_mn2ws_data *adap		Bus Info.
 *		unsigned long reg					Address Offset
 *
 * OUTPUT		:
 *		Nothing
 *
 * RETURN		:
 *		unsigned long						register value
 *
#endif
 *****************************************************************************/
static unsigned long
i2c_mn2ws_read32(struct i2c_algo_mn2ws_data *adap, unsigned long reg)
{
	void *addr;
	unsigned long val, flags;

//	DEB3( "\n\n##** %s, %d, %s **##\n\n", __FILE__, __LINE__, __func__ );
	if (!adap)
	{
		printk(KERN_ERR "i2c-mn2ws: Access to invalid pointer.\n");
		return 0;
	}
	spin_lock_irqsave(&(adap->lock), flags);
	addr = adap->iomap_base + (reg & I2C_MN2WS_RMSK);
	val  = readl(addr);	/* ioread32(addr); */
	spin_unlock_irqrestore(&(adap->lock), flags);

#ifdef DEBUG_IO
	DEB3("*** read  %s => %#04x\n", names[reg/4], val);
#endif
	return val;
}

/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		void
 *			i2c_mn2ws_add_watch_timer( void )
 *
 * DESCRIPTION	:
 *		Add watch timer
 *
 * INPUT		:
 *		struct i2c_algo_mn2ws_data *mn2ws_adap
 *
 * OUTPUT		:
 *		Nothing
 *
 * RETURN		:
 *		void
 *
#endif
 *****************************************************************************/
static void
i2c_mn2ws_add_watch_timer( struct i2c_algo_mn2ws_data *mn2ws_adap )
{
	struct i2c_algo_mn2ws_exec_info *exec_info = &mn2ws_adap->exec_info;

	DEB3( "\n\n##** %s, %d, %s **##\n\n", __FILE__, __LINE__, __func__ );
	exec_info->watch_timeflag = 1;
	exec_info->cnt_bak = exec_info->cnt;

	exec_info->watch_timer.expires = jiffies + exec_info->watch_timeval;
	exec_info->watch_timer.data = (unsigned long)mn2ws_adap;
	exec_info->watch_timer.function = i2c_mn2ws_watch_timer_handler;
	add_timer( &exec_info->watch_timer );
}

/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		void
 *			i2c_mn2ws_del_watch_timer( void )
 *
 * DESCRIPTION	:
 *		del watch timer
 *
 * INPUT		:
 *		struct i2c_algo_mn2ws_data *mn2ws_adap
 *
 * OUTPUT		:
 *		Nothing
 *
 * RETURN		:
 *		void
 *
#endif
 *****************************************************************************/
static void
i2c_mn2ws_del_watch_timer( struct i2c_algo_mn2ws_data *mn2ws_adap )
{
	struct i2c_algo_mn2ws_exec_info *exec_info = &mn2ws_adap->exec_info;

	DEB3( "\n\n##** %s, %d, %s **##\n\n", __FILE__, __LINE__, __func__ );
	exec_info->watch_timeflag = 0;
	del_timer( &exec_info->watch_timer );
}

/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		void
 *			i2c_mn2ws_watch_timer_handler( unsigned long data )
 *
 * DESCRIPTION	:
 *		Handler for timeout
 *
 * INPUT		:
 *		unsigned long data
 *
 * OUTPUT		:
 *		Nothing
 *
 * RETURN		:
 *		void
 *
#endif
 *****************************************************************************/
static void
i2c_mn2ws_watch_timer_handler( unsigned long data )
{
	struct i2c_algo_mn2ws_data *mn2ws_adap = (struct i2c_algo_mn2ws_data *)data;
	struct i2c_algo_mn2ws_exec_info *exec_info = &(mn2ws_adap->exec_info);
	unsigned long flags;

	DEB3( "\n\n##** %s, %d, %s **##\n\n", __FILE__, __LINE__, __func__ );
	exec_info->watch_timeflag = 0;

	/* check whether to have already done wakeup.  */
	if( 1 == mn2ws_adap->irq_event )
	{
		/* Wakeup has already been executed.  */
		return;
	}

	if( exec_info->cnt != exec_info->cnt_bak )
	{
		/* next watch timer */
		i2c_mn2ws_add_watch_timer( mn2ws_adap );
	}
	else {
		/* timeout error, wakeup */
		exec_info->result = -ETIMEDOUT;

		/* wake up */
		spin_lock_irqsave(&(mn2ws_adap->lock), flags);
		mn2ws_adap->irq_event = 1;
		spin_unlock_irqrestore(&(mn2ws_adap->lock), flags);
		wake_up(&(mn2ws_adap->wait));
	}
}

/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		int i2c_mn2ws_write_cmd(
 *			struct i2c_algo_mn2ws_data *adap, unsigned long reg
 *			, unsigned long val, int timeout
 *		)
 *
 * DESCRIPTION	:
 *		Write value to control register
 *		 and wait for Ack/NoAck Interrupts
 *
 * INPUT		:
 *		struct i2c_algo_mn2ws_data *adap		Bus Info.
 *		unsigned long reg					Address Offset
 *		unsigned long val					Write Value
 *		int timeout							Timeout msec [jiffies count]
 *
 * OUTPUT		:
 *		Nothing
 *
 * RETURN		:
 *		int		Error Code
 *				0		Normal
 *				< 0		Error
 *						-ENODEV    : Device Error
 *						-ETIMEDOUT : Time Out Error
 *
#endif
 *****************************************************************************/
static int
i2c_mn2ws_write_cmd(
	struct i2c_algo_mn2ws_data *adap, unsigned long reg
	, unsigned long val, int timeout
)
{
	void *addr;
	int irq;
	int ret = 0;
	unsigned long flags;

	DEB3( "\n\n##** %s, %d, %s **##\n\n", __FILE__, __LINE__, __func__ );
	if (!adap)
	{
		printk(KERN_ERR "i2c-mn2ws: Access to invalid pointer.\n");
		return -ENODEV;
	}
	irq = adap->irq;
	addr = adap->iomap_base + (reg & I2C_MN2WS_RMSK);

#ifdef DEBUG_IO
	DEB2(
		"*** Write CMD and Wait IRQ [%lu msec] TIMEOUT[%lu]\n"
		, ((jiffies * 1000) / HZ), timeout
	);
	// INTC : Enable
	DEB3("=== Enable IRQ\n");
#endif
	enable_irq(irq);

	spin_lock_irqsave(&(adap->lock), flags);
	adap->irq_event = 0;
	adap->exec_info.result = 0;
    writel(val, addr);	/* iowrite32(val, addr); */
	spin_unlock_irqrestore(&(adap->lock), flags);

#ifdef DEBUG_IO
	DEB3("*** write %s at %#lx <= %#04x\n", names[reg/4], addr, val);
#endif

	if (irq > -1)
	{
		adap->exec_info.watch_timeval = timeout;
		i2c_mn2ws_add_watch_timer( adap );

		wait_event( adap->wait, (adap->irq_event == 1) );
		ret = adap->exec_info.result;

		if( adap->exec_info.watch_timeflag == 1 )
		{
			i2c_mn2ws_del_watch_timer( adap );
		}
#ifdef DEBUG_IO
		DEB3("*** i2c_mn2ws_write_cmd Result = %d(%x)\n", ret, ret);
#endif
	}
	else
	{
		unsigned long start;

		ret = 0;
		start = jiffies;
		/* Wait for SCL High */
		while (!(i2c_mn2ws_read32(adap, I2C_MN2WS_BSTS) & I2C_MN2WS_BSTS_SCL))
		{
			if (time_after_eq(jiffies, start + timeout)) {
				if( i2c_mn2ws_read32(adap, I2C_MN2WS_BSTS) & I2C_MN2WS_BSTS_SCL ){
					break;
				}
				ret = -ETIMEDOUT;
				break;
			}
			cond_resched();
		}
	}

#ifdef DEBUG_IO
	DEB2(
		"*** End Wait [%lu msec] errno = %d\n"
		, ((jiffies * 1000) / HZ), ret
	);
	// INTC : Disable
	DEB3("=== Disable IRQ\n");
#endif
	disable_irq(irq);

	return ret;
}

/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		void i2c_mn2ws_set_busrate(
 *			struct i2c_algo_mn2ws_data *adap, unsigned int rate
 *		)
 *
 * DESCRIPTION	:
 *		Set BusRate Setting
 *
 * INPUT		:
 *		struct i2c_algo_mn2ws_data *adap	Bus Info.
 *		unsigned int rate						BusRate
 *
 * OUTPUT		:
 *		Nothing
 *
 * RETURN		:
 *		void
 *
#endif
 *****************************************************************************/
static void
i2c_mn2ws_set_busrate(struct i2c_algo_mn2ws_data *adap, unsigned int rate)
{
	unsigned long pls, lw, hold;

	DEB3( "\n\n##** %s, %d, %s **##\n\n", __FILE__, __LINE__, __func__ );
	if (!adap)
	{
		printk(KERN_ERR "i2c-mn2ws: Access to invalid pointer.\n");
		return;
	}

	/* Keep bus clock frequency (rate = SCL freq[kHz] / 10[kHz])       */
	/*  SCL freq[kHz]                                                  */
	/*      = (IOCLK) / (IIC_CLK(PLS) + (IIC_NOISE(WIDTH) - 1) / 2 + 4)*/
	/*  pls = MN10300_IOCLK / (rate x 10 x 1000) - (width - 1) / 2 - 4 */
	/* [Note] Bus clock can be changed by only attach entry.           */
	if ((rate < 1) || (40 < rate))
	{
		rate = 10;	/* Default is 100[kHz] */
	}

	i2c_mn2ws_write32(
		adap, I2C_MN2WS_BRST, I2C_MN2WS_BRST_BRST
	);	/* Clear Bus Reset */

	pls = MN_IOCLK / (rate * 10 * 1000) - (I2C_MN2WS_NOISE_WIDTH_CNT - 1) / 2 - 4;

	if (0xFFFF < pls)
	{
		pls = 0x0000FFFF;
	}
	lw = pls * I2C_MN2WS_LOW_DUTY;

    /* I2C specification:                       */
    /*  LOW period of the SCL clock value       */
    /*   Standard-mode: 4.7 MIN                 */
    /*   Fast-mode:     1.3 MIN                 */
	if(lw < 160){
		lw = 160;
	}

	i2c_mn2ws_write32(adap, I2C_MN2WS_CLK, lw << 16 | pls);

	/* Bus hold time                                          */
	/*  It is a hold time of SDA to SCL. It must be >= 320ns. */
	/*  [HOLD-TIME] = [HOLDCNT] / [IOCLK]                     */
	/*  [HOLDCNT]   = [HOLD-TIME] * [IOCLK]                   */
	/*                   [10^-9s]   [10^6s]                   */
	hold = ((unsigned long long)I2C_MN2WS_HOLD_TIME * MN_IOCLK) / (1000000000U);
	i2c_mn2ws_write32(adap, I2C_MN2WS_HOLD, hold);

	/* Noise reduction sampling                               */
	/*  It is a sampling time of noise filter of SDA and SCL. */
	/*  LSB must be "1".                                      */
	i2c_mn2ws_write32(adap, I2C_MN2WS_NOISE, I2C_MN2WS_NOISE_WIDTH_CNT);

	/* Setup time setting (default:0x20) */
	/* Setup Time = SCL * (SETUPCNT + 2) */
	i2c_mn2ws_write32(adap, I2C_MN2WS_SETUP, I2C_MN2WS_SETUPCNT);

	/* Bus Enable */
	i2c_mn2ws_write32(
		adap, I2C_MN2WS_BRST
		, I2C_MN2WS_BRST_FOEN | I2C_MN2WS_BRST_BRST
	);

	#ifdef DBG_I2C
		{
			unsigned long drec;
			unsigned long bsts;
			
			drec = i2c_mn2ws_read32( adap, I2C_MN2WS_DREC );
			if( !(drec & I2C_MN2WS_DREC_BB) ){
				printk( "%d: %s: \x1b[31mERROR\x1b[30m:\n", current->pid, __func__ );
			}
			bsts = i2c_mn2ws_read32( adap, I2C_MN2WS_BSTS );
			if( !(I2C_MN2WS_BSTS_SDA & bsts) || !(I2C_MN2WS_BSTS_SCL & bsts) ){
				printk( "%d: %s: \x1b[31mERROR\x1b[30m:\n", current->pid, __func__ );
			}
		}
	#endif	//DBG_I2C
	/* Calc Delay [sec = 1 / Hz] => [msec =    1 / kHz] */
	/*                           => [usec = 1000 / kHz] */
	adap->udelay = 1000 / (rate * 10) + 1;

#ifdef DEBUG_IO
	DEB3("*** i2c_mn2ws_set_busrate : udelay %d\n", adap->udelay);
#endif
}

/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		unsigned int i2c_mn2ws_get_busrate(struct i2c_algo_mn2ws_data *adap)
 *
 * DESCRIPTION	:
 *		Get BusRate Setting
 *
 * INPUT		:
 *		struct i2c_algo_mn2ws_data *adap		Bus Info.
 *
 * OUTPUT		:
 *		Nothing
 *
 * RETURN		:
 *		unsigned int						BusRate
 *
#endif
 *****************************************************************************/
static unsigned int
i2c_mn2ws_get_busrate(struct i2c_algo_mn2ws_data *adap)
{
	unsigned int rate;
	unsigned long pls, width;

	DEB3( "\n\n##** %s, %d, %s **##\n\n", __FILE__, __LINE__, __func__ );
	if (!adap)
	{
		printk(KERN_ERR "i2c-mn2ws: Access to invalid pointer.\n");
		return -ENODEV;
	}
	pls    = i2c_mn2ws_read32(adap, I2C_MN2WS_CLK);
	pls   &= I2C_MN2WS_CLK_PLS;
	width  = i2c_mn2ws_read32(adap, I2C_MN2WS_NOISE);
	width &= I2C_MN2WS_NOISE_WIDTH;

	rate = (MN_IOCLK / (pls + (width - 1) / 2 + 4)) / (10 * 1000);

#ifdef DEBUG_IO
	DEB3(
		"*** i2c_mn2ws_get_busrate : Rate = %x, PLS = %x, Width = %x\n"
		, rate, pls, width
	);
#endif
	return rate;
}

/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		irqreturn_t i2c_mn2ws_handler(
 *			int this_irq, void *dev_id, struct pt_regs *regs
 *		)
 *
 * DESCRIPTION	:
 *		Handler for Ack/NoAck Interrupt
 *
 * INPUT		:
 *		int this_irq
 *		void *dev_id
 *		struct pt_regs *regs
 *
 * OUTPUT		:
 *		Nothing
 *
 * RETURN		:
 *		irqreturn_t
 *
#endif
 *****************************************************************************/
static irqreturn_t
i2c_mn2ws_handler( int this_irq, void *dev_id )
{
	struct i2c_algo_mn2ws_data *mn2ws_adap =
		((struct i2c_adapter *)dev_id)->algo_data;
	struct i2c_algo_mn2ws_exec_info *exec_info =
		&(mn2ws_adap->exec_info);
	unsigned char *buf = exec_info->buf;
	unsigned long flags;
	unsigned long drec;
	unsigned long val;
	void *addr;

	DEB3( "\n\n##** %s, %d, %s(mode=%d) **##\n\n", __FILE__, __LINE__, __func__, exec_info->mode);
	/* check whether to have already done wakeup.  */
	if( 1 == mn2ws_adap->irq_event )
	{
		/* Wakeup has already been executed.  */
		return IRQ_HANDLED;
	}

#if defined(CONFIG_I2C_SLAVE)
    if( ((struct i2c_adapter *)dev_id)->attr != I2C_ATTR_SLAVE )
    {
#endif //defined(CONFIG_I2C_SLAVE)
	/* check the execution result */
	switch( exec_info->mode ) {
	case I2C_MN2WS_EXEC_START:
		drec = i2c_mn2ws_read32( mn2ws_adap, I2C_MN2WS_DREC );
		if( !(drec & I2C_MN2WS_DREC_STATUS) || (exec_info->flags & I2C_M_IGNORE_NAK) )
		{
			/* set next operation */
			if( exec_info->flags & I2C_M_RD)
			{
				exec_info->mode = I2C_MN2WS_EXEC_READ;
			}
			else {
				exec_info->mode = I2C_MN2WS_EXEC_SEND;
			}
		}
		else {
			#ifdef DBG_I2C
				printk( "%d: %s: \x1b[31mERROR\x1b[30m: %d: drec = 0x%08lX: flags = 0x%08lX\n", current->pid, __func__, __LINE__, drec, exec_info->flags );
			#endif	//DBG_I2C
			exec_info->result = -EREMOTEIO;	/* NAK */
		}
		break;

	case I2C_MN2WS_EXEC_READ:
		drec = i2c_mn2ws_read32( mn2ws_adap, I2C_MN2WS_DREC );
		buf[exec_info->cnt] = (drec & I2C_MN2WS_DREC_DATA);
		exec_info->cnt++;
		break;

	case I2C_MN2WS_EXEC_SEND:
		drec = i2c_mn2ws_read32( mn2ws_adap, I2C_MN2WS_DREC );
		if( !(drec & I2C_MN2WS_DREC_STATUS) || (exec_info->flags & I2C_M_IGNORE_NAK) )
		{
			exec_info->cnt++;
		}
		else {
			#ifdef DBG_I2C
				printk( "%d: %s: \x1b[31mERROR\x1b[30m: %d: drec = 0x%08lX: flags = 0x%08X\n", current->pid, __func__, __LINE__, drec, exec_info->flags );
			#endif	//DBG_I2C
			exec_info->result = -EREMOTEIO;	/* NAK */
		}
		break;

	default:
		break;
	}

	if( (exec_info->cnt < exec_info->len ) && ( 0 == exec_info->result ) )
	{
		/* execute the next byte */
		if( exec_info->mode == I2C_MN2WS_EXEC_READ )
		{
			if( (exec_info->cnt < (exec_info->len - 1)) && !(exec_info->flags & I2C_M_NO_RD_ACK) )
			{
				val = I2C_MN2WS_DTRM_IRQEN;				/* IRQ Enable, Send Ack */
			}
			else {
				val = I2C_MN2WS_DTRM_IRQEN						/* IRQ Enable */
					 | I2C_MN2WS_DTRM_ACK;						/* NoAck */
			}
			addr = mn2ws_adap->iomap_base + (I2C_MN2WS_DTRM & I2C_MN2WS_RMSK);
			writel( val, addr );							/* iowrite32(val, addr); */
		}
		else if( exec_info->mode == I2C_MN2WS_EXEC_SEND )
		{
			val = I2C_MN2WS_DTRM_IRQEN							/* IRQ Enable */
				 | I2C_MN2WS_DTRM_ACK							/* NOP */
				 | (I2C_MN2WS_DTRM_DATA & buf[exec_info->cnt]);	/* Send Data */
			addr = mn2ws_adap->iomap_base + (I2C_MN2WS_DTRM & I2C_MN2WS_RMSK);
			writel( val, addr );							/* iowrite32(val, addr); */
		}
	}
	else {
		/* wake up */
		spin_lock_irqsave(&(mn2ws_adap->lock), flags);
		mn2ws_adap->irq_event = 1;
		spin_unlock_irqrestore(&(mn2ws_adap->lock), flags);
		wake_up(&(mn2ws_adap->wait));
	}
#if defined(CONFIG_I2C_SLAVE)
    }
    else {
	/* slave mode */
	/* check the execution result */
	switch( exec_info->mode ) {
		case I2C_MN2WS_EXEC_SLAVE_RECV:
			drec = i2c_mn2ws_read32( mn2ws_adap, I2C_MN2WS_DREC );
			
			if(drec & I2C_MN2WS_DREC_STS )
			{
				exec_info->result = exec_info->cnt;
				exec_info->slave_cond = I2C_SLAVE_COND_STOP;
				
				/* wake up */
				mn2ws_adap->irq_event = 1;
				wake_up(&(mn2ws_adap->wait));
				
				return IRQ_HANDLED;
			}
			else if(drec & I2C_MN2WS_DREC_AAS){
				if(drec & 1){
					//write to read, exec-mode change and wake up.
					exec_info->result = exec_info->cnt;
					exec_info->slave_cond = I2C_SLAVE_COND_READ;
				}else{
					// write to write,exec-mode keep and wake up
					exec_info->result = exec_info->cnt;
					exec_info->slave_cond = I2C_SLAVE_COND_WRITE;
				}
			   	
				/* wake up */
				mn2ws_adap->irq_event = 1;
				wake_up(&(mn2ws_adap->wait));
					
				return IRQ_HANDLED;
			}
			else
			{
				buf[exec_info->cnt] = (drec & I2C_MN2WS_DREC_DATA);
				exec_info->cnt++;
			}
			
			break;

		case I2C_MN2WS_EXEC_SLAVE_SEND:
			drec = i2c_mn2ws_read32( mn2ws_adap, I2C_MN2WS_DREC );
			if( !(drec & I2C_MN2WS_DREC_STATUS) || (exec_info->flags & I2C_M_IGNORE_NAK) )
			{
				exec_info->cnt++;
			}
			else {
				if(drec & I2C_MN2WS_DREC_STS )
				{
					exec_info->cnt++;
					exec_info->result = exec_info->cnt;
					exec_info->slave_cond = I2C_SLAVE_COND_STOP;
					
					/* wake up */
					mn2ws_adap->irq_event = 1;
					wake_up(&(mn2ws_adap->wait));
					
					return IRQ_HANDLED;
				}
				else if( drec & I2C_MN2WS_DREC_LRB )
				{
					exec_info->slave_cond = I2C_SLAVE_COND_STOP;
				}
				else if(drec & I2C_MN2WS_DREC_AAS){
					if(drec & 1){
						//read to read, exec-mode keep and wake up.
						exec_info->cnt++;
						exec_info->result = exec_info->cnt;
						exec_info->slave_cond = I2C_SLAVE_COND_READ;
					}else{
						// read to write, exec-mode change and wake up.
						exec_info->cnt++;
						exec_info->result = exec_info->cnt;
						exec_info->slave_cond = I2C_SLAVE_COND_WRITE;
					}
				   	
					/* wake up */
					mn2ws_adap->irq_event = 1;
					wake_up(&(mn2ws_adap->wait));
						
					return IRQ_HANDLED;
				}
				else
				{
#ifdef DBG_I2C
					printk( "%d: %s: \x1b[31mERROR\x1b[30m: %d: drec = 0x%08lX: flags = 0x%08X\n", current->pid, __func__, __LINE__, drec, exec_info->flags );
#endif	//DBG_I2C
					exec_info->result = -EREMOTEIO;	/* NAK */
				}
			}
			break;

		case I2C_MN2WS_EXEC_SLAVE_WAIT:
			drec = i2c_mn2ws_read32( mn2ws_adap, I2C_MN2WS_DREC );
			if((drec & I2C_MN2WS_DREC_AAS) ||
			   ((drec & (I2C_MN2WS_DREC_DATA & ~1)) == 0)){
				if(drec & 1){
					//send
					mn2ws_adap->slave_event = I2C_SLAVE_REQ_READ;
					exec_info->mode = I2C_MN2WS_EXEC_SLAVE_SEND;
				}else{
					//recv
					mn2ws_adap->slave_event = I2C_SLAVE_REQ_WRITE;
					exec_info->mode = I2C_MN2WS_EXEC_SLAVE_RECV;
				}
			   	
				wake_up_interruptible( &(mn2ws_adap->slave_wait) );
				return IRQ_HANDLED;
			}
			break;
		default:
			return IRQ_HANDLED;
	}
	
	/* slave mode */
	if( (exec_info->cnt < exec_info->len ) && ( 0 == exec_info->result ) )
	{
		if( exec_info->mode == I2C_MN2WS_EXEC_SLAVE_RECV )
		{
			if( exec_info->slave_cond == I2C_SLAVE_COND_CONTINUE )
			{
				val = I2C_MN2WS_DTRM_IRQEN;				/* IRQ Enable, Send Ack */
			}
			else
			{
				val = I2C_MN2WS_DTRM_IRQEN						/* IRQ Enable */
					 | I2C_MN2WS_DTRM_ACK;						/* NoAck */
			}
			addr = mn2ws_adap->iomap_base + (I2C_MN2WS_DTRM & I2C_MN2WS_RMSK);
			writel( val, addr );
		}
		else if( exec_info->mode == I2C_MN2WS_EXEC_SLAVE_SEND )
		{
			if( exec_info->slave_cond == I2C_SLAVE_COND_CONTINUE )
			{
				val = I2C_MN2WS_DTRM_IRQEN							/* IRQ Enable */
					 | I2C_MN2WS_DTRM_ACK							/* NOP */
					 | (I2C_MN2WS_DTRM_DATA & buf[exec_info->cnt]);	/* Send Data */
				addr = mn2ws_adap->iomap_base + (I2C_MN2WS_DTRM & I2C_MN2WS_RMSK);
				writel( val, addr );
			}
		}
		else
		{
			/* no process */
		}
	}
	else {		
		/* wake up */
		mn2ws_adap->irq_event = 1;
		wake_up(&(mn2ws_adap->wait));
	}
    }
#endif//defined(CONFIG_I2C_SLAVE)

	return IRQ_HANDLED;
}

/******************************************************************************
 * Data
 *****************************************************************************/
/* I2C-0(ch0) Data */
static struct i2c_algo_mn2ws_data mn2ws_data_ch0 = {
	.base               = 0,
	.irq                = -1,
	.rate               = 0,
	.wr2rd_wait_time    = 0,
};
static struct i2c_adapter mn2ws_ops_ch0 = {
	.owner          = THIS_MODULE,
	.class          = I2C_CLASS_HWMON,
	.algo_data      = &mn2ws_data_ch0,
	.name           = "MN2WS I2C Adapter(ch0)",
	.pfControl		= i2c_mn2ws_control,
#if defined(CONFIG_I2C_SLAVE)
#if defined(CONFIG_I2C_SLAVE_CH0_ATTR)
	.attr			= I2C_ATTR_SLAVE,
#if defined(CONFIG_I2C_SLAVE_CH0_ADDR)
	.addr			= CONFIG_I2C_SLAVE_CH0_ADDR,
#endif //defined(CONFIG_I2C_SLAVE_CH0_ADDR)
#else
	.attr			= I2C_ATTR_MASTER,
#endif //defined(CONFIG_I2C_SLAVE_CH0_ATTR)
#endif //defined(CONFIG_I2C_SLAVE)
};

/* I2C-1(ch1) Data */
static struct i2c_algo_mn2ws_data mn2ws_data_ch1 = {
	.base               = 0,
	.irq                = -1,
	.rate               = 0,
	.wr2rd_wait_time    = 0,
};
static struct i2c_adapter mn2ws_ops_ch1 = {
	.owner          = THIS_MODULE,
	.class          = I2C_CLASS_HWMON,
	.algo_data      = &mn2ws_data_ch1,
	.name           = "MN2WS I2C Adapter(ch1)",
	.pfControl		= i2c_mn2ws_control,
#if defined(CONFIG_I2C_SLAVE)
#if defined(CONFIG_I2C_SLAVE_CH1_ATTR)
	.attr			= I2C_ATTR_SLAVE,
#if defined(CONFIG_I2C_SLAVE_CH1_ADDR)
	.addr			= CONFIG_I2C_SLAVE_CH1_ADDR,
#endif //defined(CONFIG_I2C_SLAVE_CH1_ADDR)
#else
	.attr			= I2C_ATTR_MASTER,
#endif //defined(CONFIG_I2C_SLAVE_CH1_ATTR)
#endif //defined(CONFIG_I2C_SLAVE)
};

/* I2C-1(ch2) Data */
static struct i2c_algo_mn2ws_data mn2ws_data_ch2 = {
	.base               = 0,
	.irq                = -1,
	.rate               = 0,
	.wr2rd_wait_time    = 0,
};
static struct i2c_adapter mn2ws_ops_ch2 = {
	.owner          = THIS_MODULE,
	.class          = I2C_CLASS_HWMON,
	.algo_data      = &mn2ws_data_ch2,
	.name           = "MN2WS I2C Adapter(ch2)",
	.pfControl		= i2c_mn2ws_control,
#if defined(CONFIG_I2C_SLAVE)
#if defined(CONFIG_I2C_SLAVE_CH2_ATTR)
	.attr			= I2C_ATTR_SLAVE,
#if defined(CONFIG_I2C_SLAVE_CH2_ADDR)
	.addr			= CONFIG_I2C_SLAVE_CH2_ADDR,
#endif //defined(CONFIG_I2C_SLAVE_CH2_ADDR)
#else
	.attr			= I2C_ATTR_MASTER,
#endif //defined(CONFIG_I2C_SLAVE_CH2_ATTR)
#endif //defined(CONFIG_I2C_SLAVE)
};

/* I2C-1(ch3) Data */
static struct i2c_algo_mn2ws_data mn2ws_data_ch3 = {
	.base               = 0,
	.irq                = -1,
	.rate               = 0,
	.wr2rd_wait_time    = 0,
};
static struct i2c_adapter mn2ws_ops_ch3 = {
	.owner          = THIS_MODULE,
	.class          = I2C_CLASS_HWMON,
	.algo_data      = &mn2ws_data_ch3,
	.name           = "MN2WS I2C Adapter(ch3)",
	.pfControl		= i2c_mn2ws_control,
#if defined(CONFIG_I2C_SLAVE)
#if defined(CONFIG_I2C_SLAVE_CH3_ATTR)
	.attr			= I2C_ATTR_SLAVE,
#if defined(CONFIG_I2C_SLAVE_CH3_ADDR)
	.addr			= CONFIG_I2C_SLAVE_CH3_ADDR,
#endif //defined(CONFIG_I2C_SLAVE_CH3_ADDR)
#else
	.attr			= I2C_ATTR_MASTER,
#endif //defined(CONFIG_I2C_SLAVE_CH3_ATTR)
#endif //defined(CONFIG_I2C_SLAVE)
};

/* I2C-1(ch4) Data */
static struct i2c_algo_mn2ws_data mn2ws_data_ch4 = {
	.base               = 0,
	.irq                = -1,
	.rate               = 0,
	.wr2rd_wait_time    = 0,
};
static struct i2c_adapter mn2ws_ops_ch4 = {
	.owner          = THIS_MODULE,
	.class          = I2C_CLASS_HWMON,
	.algo_data      = &mn2ws_data_ch4,
	.name           = "MN2WS I2C Adapter(ch4)",
	.pfControl		= i2c_mn2ws_control,
#if defined(CONFIG_I2C_SLAVE)
#if defined(CONFIG_I2C_SLAVE_CH4_ATTR)
	.attr			= I2C_ATTR_SLAVE,
#if defined(CONFIG_I2C_SLAVE_CH4_ADDR)
	.addr			= CONFIG_I2C_SLAVE_CH4_ADDR,
#endif //defined(CONFIG_I2C_SLAVE_CH4_ADDR)
#else
	.attr			= I2C_ATTR_MASTER,
#endif //defined(CONFIG_I2C_SLAVE_CH4_ATTR)
#endif //defined(CONFIG_I2C_SLAVE)
};

static struct i2c_adapter *mn2ws_ops[I2C_MN2WS_ADAP_MAX + 1] = {
	[0] = &mn2ws_ops_ch0,
	[1] = &mn2ws_ops_ch1,
	[2] = &mn2ws_ops_ch2,
	[3] = &mn2ws_ops_ch3,
	[4] = &mn2ws_ops_ch4,
	[I2C_MN2WS_ADAP_MAX] = 0,
};

/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		int __devinit i2c_mn2ws_bus_init_sub(struct i2c_adapter *i2c_adap)
 *
 * DESCRIPTION	:
 *		Sub-function for i2c_mn2ws_bus_init()
 *
 * INPUT		:
 *		struct i2c_adapter *i2c_adap	Adapter Info.
 *
 * OUTPUT		:
 *		Nothing
 *
 * RETURN		:
 *		int		Error Code
 *				0		Normal
 *				< 0		Error
 *
#endif
 *****************************************************************************/
static int
__devinit i2c_mn2ws_bus_init_sub(struct i2c_adapter *i2c_adap)
{
	struct i2c_algo_mn2ws_data *mn2ws_adap;
	unsigned long base;
	int           irq;

	DEB3( "\n\n##** %s, %d, %s **##\n\n", __FILE__, __LINE__, __func__ );
	if (!i2c_adap)
	{
		printk(KERN_ERR "i2c-mn2ws: Access to invalid pointer.\n");
		return -ENODEV;
	}
	mn2ws_adap = i2c_adap->algo_data;

	if (!mn2ws_adap)
	{
		printk(KERN_ERR "i2c-mn2ws: Access to invalid pointer.\n");
		return -ENODEV;
	}
	base = mn2ws_adap->base;
	irq  = mn2ws_adap->irq;

	init_waitqueue_head(&(mn2ws_adap->wait));
	mn2ws_adap->irq_event = 0;
	spin_lock_init(&(mn2ws_adap->lock));

#if defined(CONFIG_I2C_SLAVE)
	init_waitqueue_head(&(mn2ws_adap->slave_wait));
#endif//defined(CONFIG_I2C_SLAVE)
	
	printk(KERN_INFO "i2c-mn2ws: i/o base %#08lx. irq %d\n", base, irq);

	if (!request_mem_region(base, IO_SIZE, "i2c-mn2ws"))
	{
		printk(KERN_ERR "i2c-mn2ws: I/O address %#08lx is in use.\n", base);
		goto out;
	}

	mn2ws_adap->iomap_base = ioremap(base, IO_SIZE);
	if (!(mn2ws_adap->iomap_base))
	{
		printk(KERN_ERR "i2c-mn2ws: I/O address %#08lx can't map.\n", base);
		goto out_region;
	}

	disable_irq(irq);

	if (irq > -1)
	{
		if (request_irq(irq, i2c_mn2ws_handler, 0, "i2c-mn2ws", i2c_adap) < 0)
		{
			printk(KERN_ERR "i2c-mn2ws: Request irq%d failed\n", irq);
			goto out_remap;
		}
	}
	// Clear INTC
	mn_intc_clear( irq );
#ifdef DEBUG_IO
	DEB3("GxICR(%d) = %x\n", irq, GxICR(irq));
#endif

	disable_irq(irq);

	init_timer( &mn2ws_adap->exec_info.watch_timer );

	if (i2c_mn2ws_add_bus(i2c_adap) < 0)
	{
		printk(KERN_ERR "i2c-mn2ws: Failed to add i2c bus\n");
		goto out_irq;
	}
	return 0;

 out_irq:
	if (irq > -1)
		free_irq(irq, i2c_adap);
 out_remap:
	iounmap(mn2ws_adap->iomap_base);
 out_region:
	release_mem_region(base, IO_SIZE);
 out:
	return -ENODEV;
}

static int __devinit i2c_mn2ws_probe( struct platform_device *pdev )
{
	struct i2c_mn2ws_pdata		*pdata = pdev->dev.platform_data;
	struct i2c_algo_mn2ws_data		*psAlgoData;
	int		ret;
	
	
	if( I2C_MN2WS_ADAP_MAX <= pdev->id){
		return -EINVAL;
	}
	
	psAlgoData = (struct i2c_algo_mn2ws_data *)mn2ws_ops[pdev->id]->algo_data;
	psAlgoData->base = pdata->base;
	psAlgoData->irq  = pdata->irq;
	psAlgoData->rate = pdata->rate;
	
	ret = i2c_mn2ws_bus_init_sub( mn2ws_ops[pdev->id] );

#if defined(CONFIG_I2C_SLAVE)
	if( mn2ws_ops[pdev->id]->attr == I2C_ATTR_SLAVE )
	{
		void *addr;
		
		addr = psAlgoData->iomap_base + I2C_MN2WS_MYAD;
		writel( mn2ws_ops[pdev->id]->addr, addr );
		
		printk( KERN_INFO "i2c-mn2ws: slave_channel = %d, slave address = 0x%x\n", pdev->id, mn2ws_ops[pdev->id]->addr );
		
	}
#endif //defined(CONFIG_I2C_SLAVE)
	
	return ret;
}

static struct platform_driver i2c_mn2ws_driver = {
	.probe = i2c_mn2ws_probe,
	.remove = __devexit_p(i2c_mn2ws_remove),
	.suspend = i2c_mn2ws_suspend,
	.resume = i2c_mn2ws_resume,
	.driver = {
		.name = "i2c-mn2ws",
		.owner = THIS_MODULE,
	},
};

/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		void i2c_mn2ws_bus_exit_sub(struct i2c_adapter *i2c_adap)
 *
 * DESCRIPTION	:
 *		Sub-function for i2c_mn2ws_bus_exit()
 *
 * INPUT		:
 *		struct i2c_adapter *i2c_adap	Adapter Info.
 *
 * OUTPUT		:
 *		Nothing
 *
 * RETURN		:
 *		void
 *
#endif
 *****************************************************************************/
static void
i2c_mn2ws_bus_exit_sub(struct i2c_adapter *i2c_adap)
{
	struct i2c_algo_mn2ws_data *mn2ws_adap;
	unsigned long base;
	int           irq;

	DEB3( "\n\n##** %s, %d, %s **##\n\n", __FILE__, __LINE__, __func__ );
	if (!i2c_adap)
	{
		printk(KERN_ERR "i2c-mn2ws: Access to invalid pointer.\n");
		return;
	}
	mn2ws_adap = i2c_adap->algo_data;

	if (!mn2ws_adap)
	{
		printk(KERN_ERR "i2c-mn2ws: Access to invalid pointer.\n");
		return;
	}
	base = mn2ws_adap->base;
	irq  = mn2ws_adap->irq;

	i2c_mn2ws_del_bus(i2c_adap);

	if (irq > 0) {
		disable_irq(irq);
		free_irq(irq, i2c_adap);
	}
	iounmap(mn2ws_adap->iomap_base);
	release_mem_region(base, IO_SIZE);
}

static int __devexit i2c_mn2ws_remove( struct platform_device *pdev )
{
	DEB3( "\n\n##** %s, %d, %s **##\n\n", __FILE__, __LINE__, __func__ );
	i2c_mn2ws_bus_exit_sub( mn2ws_ops[pdev->id] );
	
	return 0;
}

static int i2c_mn2ws_suspend( struct platform_device *pdev, pm_message_t state )
{
	struct i2c_algo_mn2ws_data *mn2ws_adap = mn2ws_ops[pdev->id]->algo_data;
	int		irq;

	irq = mn2ws_adap->irq;

	disable_irq(irq);

	/* GIC interrupt disable */
	mn_intc_disable(irq);

	return 0;
}

static int i2c_mn2ws_resume( struct platform_device *pdev )
{
	struct i2c_algo_mn2ws_data *mn2ws_adap = mn2ws_ops[pdev->id]->algo_data;
	int		irq, ret;
	
	irq = mn2ws_adap->irq;

	/* GIC interrupt clear & enable */
	mn_intc_clear(irq);
	mn_intc_enable(irq);

	enable_irq(irq);

	/* hardware initialization only */
	ret = i2c_mn2ws_init(mn2ws_adap);
	
	return ret;
}

/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		int __init i2c_mn2ws_bus_init(void)
 *
 * DESCRIPTION	:
 *		Add I2C Bus Info.
 *
 * INPUT		:
 *		void
 *
 * OUTPUT		:
 *		Nothing
 *
 * RETURN		:
 *		int		Error Code
 *				0		Normal
 *				< 0		Error
 *
#endif
 *****************************************************************************/
static int __init i2c_mn2ws_bus_init( void )
{
	int		status;
	
	status = platform_driver_register( &i2c_mn2ws_driver );
	
	return status;
}

/******************************************************************************
#ifdef DOC
 * FUNCTION		:
 *		static void i2c_mn2ws_bus_exit(void)
 *
 * DESCRIPTION	:
 *		Remove I2C Bus Info.
 *
 * INPUT		:
 *		void
 *
 * OUTPUT		:
 *		Nothing
 *
 * RETURN		:
 *		void
 *
#endif
 *****************************************************************************/
static void __exit i2c_mn2ws_bus_exit(void)
{
	platform_driver_unregister( &i2c_mn2ws_driver );
}

/******************************************************************************
 * Module Define
 *****************************************************************************/
MODULE_LICENSE("GPL");

module_init(i2c_mn2ws_bus_init);
module_exit(i2c_mn2ws_bus_exit);

module_param(i2c_debug, int, 0);
